#include <iostream>
#include <stdexcept>

// ---- Template Selection Helper
  
template<bool Condition, class T = void>
struct enable_if {
  typedef T type;
};

template<class T>
struct enable_if<false, T> {};
  
// ---- Meta Assertions

#define STATIC_ASSERT(x) static const int STATIC_ASSERT_ ## __COUNT__ [(x)]

// ---- Runtime Error

struct out_of_range : virtual std::out_of_range {
  out_of_range() : std::out_of_range("value out of range") {}
};

// ---- Our Actual 'restricted' type

template<class T, T Min, T Max>
struct restricted {
  STATIC_ASSERT((Min < Max));
  
  T value;
  
  explicit restricted(T o)
    : value(o) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
  
  template<T OMin, T OMax>
  restricted(restricted<T, OMin, OMax> o
    //
    // Our previous check only tested for part of the story.
    //
    // All compile-time constant expressions can be used with
    // numbers.  Including && or ||.
    //
     , typename enable_if< OMin <= Max && Min <= OMax>::type* = 0
  )
    : value(o.value) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
};

// ---- How to use it

int main() {
  restricted<int, 0, 10> i(5);
  
  // These must succeed:
  restricted<int, -5, 5> leftOverlap(i);
  restricted<int, 5, 15> rightOverlap(i);
  restricted<int, 3, 8> subset(i);
  
  // These must fail at compile time:
  // TODO: comment out to continue
  restricted<int, -10, -1> tooLow(i);
  restricted<int, 11, 20> tooHigh(i);
  
  // If the compile-time range _may_ fit, but during
  // runtime an out_of_range is detected, this still works:
  try {
    restricted<int, 0, 4> j(i);
    std::cerr << "ERROR: out_of_range not thrown" << std::endl;
  } catch(out_of_range const&) {}
  
  try {
    restricted<int, 6, 10> j(i);
    std::cerr << "ERROR: out_of_range not thrown" << std::endl;
  } catch(out_of_range const&) {}
}
